<?php
function hasPermission($permission) {
    if (!isset($_SESSION['user_id'])) {
        error_log("No user_id in session");
        return false;
    }

    global $pdo;
    
    try {
        // First check if user is Super Admin
        $stmt = $pdo->prepare("
            SELECT r.is_super_admin 
            FROM roles r
            JOIN users u ON r.id = u.role_id
            WHERE u.id = ?
        ");
        $stmt->execute([$_SESSION['user_id']]);
        $isSuperAdmin = $stmt->fetchColumn();
        
        // If user is Super Admin, grant all permissions
        if ($isSuperAdmin) {
            error_log("User is Super Admin - granting all permissions");
            return true;
        }

        // Otherwise check specific permission
        $stmt = $pdo->prepare("
            SELECT COUNT(*) 
            FROM role_permissions rp
            JOIN permissions p ON rp.permission_id = p.id
            WHERE rp.role_id = (
                SELECT role_id 
                FROM users 
                WHERE id = ?
            )
            AND p.name = ?
        ");
        $stmt->execute([$_SESSION['user_id'], $permission]);
        $hasPermission = $stmt->fetchColumn() > 0;
        error_log("Checking permission '$permission' for user {$_SESSION['user_id']}: " . ($hasPermission ? 'Yes' : 'No'));
        return $hasPermission;
    } catch (PDOException $e) {
        error_log("Error checking permission: " . $e->getMessage());
        return false;
    }
}

function getUserPermissions() {
    if (!isset($_SESSION['user_id'])) {
        return [];
    }

    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT TRIM(p.name) as name
            FROM role_permissions rp
            JOIN permissions p ON rp.permission_id = p.id
            WHERE rp.role_id = (
                SELECT role_id 
                FROM users 
                WHERE id = ?
            )
        ");
        $stmt->execute([$_SESSION['user_id']]);
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (PDOException $e) {
        error_log("Error getting user permissions: " . $e->getMessage());
        return [];
    }
}

function getUserRole() {
    if (!isset($_SESSION['user_id'])) {
        return null;
    }

    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT r.name
            FROM roles r
            JOIN users u ON r.id = u.role_id
            WHERE u.id = ?
        ");
        $stmt->execute([$_SESSION['user_id']]);
        return $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Error getting user role: " . $e->getMessage());
        return null;
    }
}

function requirePermission($permission) {
    if (!hasPermission($permission)) {
        header('Location: ../pages/dashboard.php');
        exit;
    }
}

function requireLogin() {
    if (!isset($_SESSION['user_id'])) {
        header('Location: ../auth/login.php');
        exit;
    }
}

/**
 * Check if user has ANY of the specified permissions
 * @param array $permissions Array of permission names
 * @return bool True if user has at least one of the permissions
 */
function hasAnyPermission($permissions) {
    if (!is_array($permissions)) {
        $permissions = [$permissions];
    }
    
    foreach ($permissions as $permission) {
        if (hasPermission($permission)) {
            return true;
        }
    }
    return false;
}

/**
 * Check if user has ALL of the specified permissions
 * @param array $permissions Array of permission names
 * @return bool True if user has all permissions
 */
function hasAllPermissions($permissions) {
    if (!is_array($permissions)) {
        $permissions = [$permissions];
    }
    
    foreach ($permissions as $permission) {
        if (!hasPermission($permission)) {
            return false;
        }
    }
    return true;
}

/**
 * Get module permissions for a user (e.g., all product-related permissions)
 * @param string $module Module name (e.g., 'product', 'customer', 'supplier')
 * @return array Array of permissions the user has for this module
 */
function getModulePermissions($module) {
    if (!isset($_SESSION['user_id'])) {
        return [];
    }

    global $pdo;
    
    try {
        // First check if user is Super Admin
        $stmt = $pdo->prepare("
            SELECT r.is_super_admin 
            FROM roles r
            JOIN users u ON r.id = u.role_id
            WHERE u.id = ?
        ");
        $stmt->execute([$_SESSION['user_id']]);
        $isSuperAdmin = $stmt->fetchColumn();
        
        if ($isSuperAdmin) {
            // Get all permissions for this module
            $stmt = $pdo->prepare("
                SELECT TRIM(p.name) as name
                FROM permissions p
                WHERE p.name LIKE ?
            ");
            $stmt->execute(["%{$module}%"]);
            return $stmt->fetchAll(PDO::FETCH_COLUMN);
        }

        // Get user's permissions for this module
        $stmt = $pdo->prepare("
            SELECT TRIM(p.name) as name
            FROM role_permissions rp
            JOIN permissions p ON rp.permission_id = p.id
            WHERE rp.role_id = (
                SELECT role_id 
                FROM users 
                WHERE id = ?
            )
            AND p.name LIKE ?
        ");
        $stmt->execute([$_SESSION['user_id'], "%{$module}%"]);
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (PDOException $e) {
        error_log("Error getting module permissions: " . $e->getMessage());
        return [];
    }
}

/**
 * Check if user can access a module (has any permission for it)
 * @param array $modulePermissions Array of permissions for the module
 * @return bool True if user has at least one permission for the module
 */
function canAccessModule($modulePermissions) {
    return hasAnyPermission($modulePermissions);
}
?> 